/*
 * Copyright (c) 2006 The Regents of the University of California.
 * Copyright (c) 2004-2006 Voltaire, Inc. All rights reserved.
 * Copyright (c) 2002-2005 Mellanox Technologies LTD. All rights reserved.
 * Copyright (c) 1996-2003 Intel Corporation. All rights reserved.
 *
 * Produced at Lawrence Livermore National Laboratory.
 * Written by Ira Weiny <weiny2@llnl.gov>.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * $Id: saquery.c 9258 2006-09-05 13:24:55Z halr $
 */

#include <stdio.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <assert.h>
#include <string.h>

#define _GNU_SOURCE
#include <getopt.h>

#include <infiniband/opensm/osm_log.h>
#include <infiniband/vendor/osm_vendor_api.h>
#include <infiniband/vendor/osm_vendor_sa_api.h>
#include <infiniband/opensm/osm_mad_pool.h>
#include <infiniband/complib/cl_debug.h>

static char *argv0 = "saquery";

/**
 * Declare some globals because I don't want this to be too complex.
 */
#define MAX_PORTS (8)
osmv_query_res_t   result;
osm_log_t          log_osm;
osm_mad_pool_t     mad_pool;
osm_vendor_t      *vendor = NULL;
int                osm_debug = 0;

enum {
	ALL,
	LID_ONLY,
	GUID_ONLY,
} node_print_desc = ALL;

char              *requested_name = NULL;
ib_net16_t         query_type = IB_MAD_ATTR_NODE_RECORD;

/**
 * Call back for the various record requests.
 */
static void
query_res_cb(osmv_query_res_t *res)
{
	result = *res;
}

static void
print_node_record(ib_node_record_t *node_record)
{
	ib_node_info_t *p_ni = NULL;
	p_ni = &(node_record->node_info);
        
	switch (node_print_desc) {
	case LID_ONLY:
		printf("%d\n", cl_ntoh16(node_record->lid));
		return;
	case GUID_ONLY:
		printf("0x%016" PRIx64 "\n", cl_ntoh64(p_ni->port_guid));
		return;
	case ALL:
	default:
		break;
        }

	printf("NodeRecord dump:\n"
	       "\t\t\t\tlid.....................0x%X\n"
	       "\t\t\t\treserved................0x%X\n"
	       "\t\t\t\tbase_version............0x%X\n"
	       "\t\t\t\tclass_version...........0x%X\n"
	       "\t\t\t\tnode_type...............%s\n"
	       "\t\t\t\tnum_ports...............0x%X\n"
	       "\t\t\t\tsys_guid................0x%016" PRIx64 "\n"
	       "\t\t\t\tnode_guid...............0x%016" PRIx64 "\n"
	       "\t\t\t\tport_guid...............0x%016" PRIx64 "\n"
	       "\t\t\t\tpartition_cap...........0x%X\n"
	       "\t\t\t\tdevice_id...............0x%X\n"
	       "\t\t\t\trevision................0x%X\n"
	       "\t\t\t\tport_num................0x%X\n"
	       "\t\t\t\tvendor_id...............0x%X\n"
	       "\t\t\t\tNodeDescription.........%s\n"
	       "",
	       cl_ntoh16(node_record->lid),
	       cl_ntoh16(node_record->resv),
	       p_ni->base_version,
	       p_ni->class_version,
	       ib_get_node_type_str( p_ni->node_type ),
	       p_ni->num_ports,
	       cl_ntoh64( p_ni->sys_guid ),
	       cl_ntoh64( p_ni->node_guid ),
	       cl_ntoh64( p_ni->port_guid ),
	       cl_ntoh16( p_ni->partition_cap ),
	       cl_ntoh16( p_ni->device_id ),
	       cl_ntoh32( p_ni->revision ),
	       ib_node_info_get_local_port_num( p_ni ),
	       cl_ntoh32( ib_node_info_get_vendor_id( p_ni )),
	       node_record->node_desc.description
	       );
}

static void
print_path_record(ib_path_rec_t *p_pr)
{
	printf("\nPathRecord dump:\n"
	       "\t\t\t\tresv0...................0x%016" PRIx64 "\n"
	       "\t\t\t\tdgid....................0x%016" PRIx64 " : "
	       "0x%016" PRIx64 "\n"
	       "\t\t\t\tsgid....................0x%016" PRIx64 " : "
	       "0x%016" PRIx64 "\n"
	       "\t\t\t\tdlid....................0x%X\n"
	       "\t\t\t\tslid....................0x%X\n"
	       "\t\t\t\thop_flow_raw............0x%X\n"
	       "\t\t\t\ttclass..................0x%X\n"
	       "\t\t\t\tnum_path_revers.........0x%X\n"
	       "\t\t\t\tpkey....................0x%X\n"
	       "\t\t\t\tsl......................0x%X\n"
	       "\t\t\t\tmtu.....................0x%X\n"
	       "\t\t\t\trate....................0x%X\n"
	       "\t\t\t\tpkt_life................0x%X\n"
	       "\t\t\t\tpreference..............0x%X\n"
	       "\t\t\t\tresv2...................0x%X\n"
	       "\t\t\t\tresv3...................0x%X\n"
	       "",
	       *(uint64_t*)p_pr->resv0,
	       cl_ntoh64( p_pr->dgid.unicast.prefix ),
	       cl_ntoh64( p_pr->dgid.unicast.interface_id ),
	       cl_ntoh64( p_pr->sgid.unicast.prefix ),
	       cl_ntoh64( p_pr->sgid.unicast.interface_id ),
	       cl_ntoh16( p_pr->dlid ),
	       cl_ntoh16( p_pr->slid ),
	       cl_ntoh32( p_pr->hop_flow_raw ),
	       p_pr->tclass,
	       p_pr->num_path,
	       cl_ntoh16( p_pr->pkey ),
	       cl_ntoh16( p_pr->sl ),
	       p_pr->mtu,
	       p_pr->rate,
	       p_pr->pkt_life,
	       p_pr->preference,
	       *(uint32_t*)&p_pr->resv2,
	       *((uint16_t*)&p_pr->resv2 + 2)
	       );
}

static void
print_portinfo_record(ib_portinfo_record_t *p_pir)
{
	const ib_port_info_t * const p_pi = &p_pir->port_info;

        printf("\nPortInfoRecord dump:\n"
	       "\t\t\t\tEndPortLid..............0x%X\n"
	       "\t\t\t\tPortNum.................0x%X\n"
	       "\t\t\t\tbase_lid................0x%X\n"
	       "\t\t\t\tmaster_sm_base_lid......0x%X\n"
	       "\t\t\t\tcapability_mask.........0x%X\n"
	       "",
	       cl_ntoh16(p_pir->lid),
	       p_pir->port_num,
	       cl_ntoh16( p_pi->base_lid ),
	       cl_ntoh16( p_pi->master_sm_base_lid ),
	       cl_ntoh32( p_pi->capability_mask )
               );
}

static void
print_multicast_group_record(ib_member_rec_t *p_mcmr)
{
	        printf("\nMCMemberRecord group dump:\n"
		       "\t\t\t\tMGID....................0x%016" PRIx64 " : "
		       "0x%016" PRIx64 "\n"
		       "\t\t\t\tMlid....................0x%X\n"
		       "\t\t\t\tMtu.....................0x%X\n"
		       "\t\t\t\tpkey....................0x%X\n"
		       "\t\t\t\tRate....................0x%X\n"
		       "",
		       cl_ntoh64( p_mcmr->mgid.unicast.prefix ),
		       cl_ntoh64( p_mcmr->mgid.unicast.interface_id ),
		       cl_ntoh16( p_mcmr->mlid ),
		       p_mcmr->mtu,
		       cl_ntoh16( p_mcmr->pkey ),
		       p_mcmr->rate
		       );
}

static void
print_multicast_member_record(ib_member_rec_t *p_mcmr)
{
		printf("\nMCMemberRecord member dump:\n"
		       "\t\t\t\tMGID....................0x%016" PRIx64 " : "
		       "0x%016" PRIx64 "\n"
		       "\t\t\t\tMlid....................0x%X\n"
		       "\t\t\t\tPortGid.................0x%016" PRIx64 " : "
		       "0x%016" PRIx64 "\n"
		       "\t\t\t\tScopeState..............0x%X\n"
		       "\t\t\t\tProxyJoin...............0x%X\n"
		       "",
		       cl_ntoh64( p_mcmr->mgid.unicast.prefix ),
		       cl_ntoh64( p_mcmr->mgid.unicast.interface_id ),
		       cl_ntoh16( p_mcmr->mlid ),
		       cl_ntoh64( p_mcmr->port_gid.unicast.prefix ),
		       cl_ntoh64( p_mcmr->port_gid.unicast.interface_id ),
		       p_mcmr->scope_state,
		       p_mcmr->proxy_join
		       );
}

static void
return_mad(void)
{
	/*
	 * Return the IB query MAD to the pool as necessary.
	 */
	if( result.p_result_madw != NULL ) {
		osm_mad_pool_put( &mad_pool, result.p_result_madw );
		result.p_result_madw = NULL;
	}
}

/**
 * Get all the records available for requested query type.
 */
static ib_api_status_t
get_all_records(osm_bind_handle_t bind_handle,
		ib_net16_t query_id,
		ib_net16_t attr_offset,
		int trusted)
{
	ib_api_status_t   status;
	osmv_query_req_t  req;
	osmv_user_query_t user;

	memset( &req, 0, sizeof( req ) );
	memset( &user, 0, sizeof( user ) );

	user.attr_id = query_id;
	user.attr_offset = attr_offset;

	req.query_type = OSMV_QUERY_USER_DEFINED;
	req.timeout_ms = 100;
	req.retry_cnt = 1;
	req.flags = OSM_SA_FLAGS_SYNC;
	req.query_context = NULL;
	req.pfn_query_cb = query_res_cb;
	req.p_query_input = &user;
	if (trusted)
		req.sm_key = OSM_DEFAULT_SM_KEY;
	else
		req.sm_key = 0;

	if ((status = osmv_query_sa(bind_handle, &req)) != IB_SUCCESS) {
		fprintf(stderr, "Query SA failed: %s\n",
			ib_get_err_str(status));
		return (status);
	}

	if (result.status != IB_SUCCESS) {
		fprintf(stderr, "Query result returned: %s\n",
			ib_get_err_str(result.status));
		return (result.status);
	}
	return (status);
}

/*
 * Get the portinfo records available with IsSM CapabilityMask bit on.
 */
static ib_api_status_t
get_issm_records(osm_bind_handle_t bind_handle)
{
	ib_api_status_t   status;
	osmv_query_req_t  req;
	osmv_user_query_t user;
	ib_portinfo_record_t attr;

	memset( &req, 0, sizeof( req ) );
	memset( &user, 0, sizeof( user ) );
	memset( &attr, 0, sizeof ( attr ) );
	attr.port_info.capability_mask = IB_PORT_CAP_IS_SM;

	user.attr_id = IB_MAD_ATTR_PORTINFO_RECORD;
	user.attr_offset = ib_get_attr_offset(sizeof(ib_portinfo_record_t));
	user.attr_mod = cl_ntoh32(1 << 31);	/* enhanced query */
	user.comp_mask = IB_PIR_COMPMASK_CAPMASK;
	user.p_attr = &attr;

	req.query_type = OSMV_QUERY_USER_DEFINED;
	req.timeout_ms = 100;
	req.retry_cnt = 1;
	req.flags = OSM_SA_FLAGS_SYNC;
	req.query_context = NULL;
	req.pfn_query_cb = query_res_cb;
	req.p_query_input = &user;
	req.sm_key = 0;

	if ((status = osmv_query_sa(bind_handle, &req)) != IB_SUCCESS) {
		fprintf(stderr, "Query SA failed: %s\n",
			ib_get_err_str(status));
		return (status);
	}

	if (result.status != IB_SUCCESS) {
		fprintf(stderr, "Query result returned: %s\n",
			ib_get_err_str(result.status));
		return (result.status);
	}
	return (status);
}

static ib_api_status_t
print_node_records(osm_bind_handle_t bind_handle)
{
	int               i = 0;
	ib_node_record_t *node_record = NULL;
	ib_net16_t        attr_offset = ib_get_attr_offset(sizeof(*node_record));
	ib_api_status_t   status;

	status  = get_all_records(bind_handle, IB_MAD_ATTR_NODE_RECORD, attr_offset, 0);
	if (status != IB_SUCCESS)
		return (status);

	for (i = 0; i < result.result_cnt; i++) {
		node_record = osmv_get_query_node_rec(result.p_result_madw, i);
		if (!requested_name ||
		    (strcmp(requested_name,
			    node_record->node_desc.description) == 0))
			print_node_record(node_record);
	}
	return_mad();
	return (status);
}

static ib_api_status_t
print_path_records(osm_bind_handle_t bind_handle)
{
	int               i = 0;
	ib_path_rec_t    *path_record = NULL;
	ib_net16_t        attr_offset = ib_get_attr_offset(sizeof(*path_record));
	ib_api_status_t   status;

	status = get_all_records(bind_handle, IB_MAD_ATTR_PATH_RECORD, attr_offset, 0);
	if (status != IB_SUCCESS)
		return (status);

	for (i = 0; i < result.result_cnt; i++) {
		path_record = osmv_get_query_path_rec(result.p_result_madw, i);
		print_path_record(path_record);
	}
	return_mad();
	return (status);
}

ib_api_status_t
print_portinfo_records(osm_bind_handle_t bind_handle)
{
	int                   i = 0;
	ib_portinfo_record_t *portinfo_record = NULL;
	ib_api_status_t       status;

	status = get_issm_records(bind_handle);
	if (status != IB_SUCCESS)
		return (status);

	for (i = 0; i < result.result_cnt; i++) {
		portinfo_record = osmv_get_query_portinfo_rec(result.p_result_madw, i);
		print_portinfo_record(portinfo_record);
	}
	return_mad();
	return (status);
}

static ib_api_status_t
print_multicast_group_records(osm_bind_handle_t bind_handle, int members)
{
	int               i = 0;
	ib_member_rec_t  *mcast_record = NULL;
	ib_net16_t        attr_offset = ib_get_attr_offset(sizeof(*mcast_record));
	ib_api_status_t   status;

	status = get_all_records(bind_handle, IB_MAD_ATTR_MCMEMBER_RECORD, attr_offset, members);
	if (status != IB_SUCCESS)
		return (status);

	for (i = 0; i < result.result_cnt; i++) {
		mcast_record = osmv_get_query_mc_rec(result.p_result_madw, i);
		if (members == 0)
			print_multicast_group_record(mcast_record);
		else
			print_multicast_member_record(mcast_record);
	}
	return_mad();
	return (status);
}

static osm_bind_handle_t
get_bind_handle(void)
{
	uint32_t           i = 0;
	uint64_t           port_guid = (uint64_t)-1;
	osm_bind_handle_t  bind_handle;
	ib_api_status_t    status;
	ib_port_attr_t     attr_array[MAX_PORTS];
	uint32_t           num_ports = MAX_PORTS;

	complib_init();

	osm_log_construct(&log_osm);
	if ((status = osm_log_init_v2(&log_osm, TRUE, 0x0001, NULL,
				      0, TRUE)) != IB_SUCCESS) {
		fprintf(stderr, "Failed to init osm_log: %s\n",
			ib_get_err_str(status));
		exit(-1);
	}
	osm_log_set_level(&log_osm, OSM_LOG_NONE);
	if (osm_debug)
		osm_log_set_level(&log_osm, OSM_LOG_DEFAULT_LEVEL);

        vendor = osm_vendor_new(&log_osm, 100);
	osm_mad_pool_construct(&mad_pool);
	if ((status = osm_mad_pool_init(&mad_pool, &log_osm)) != IB_SUCCESS) {
		fprintf(stderr, "Failed to init mad pool: %s\n",
			ib_get_err_str(status));
		exit(-1);
	}

	if ((status = osm_vendor_get_all_port_attr(vendor, attr_array, &num_ports)) != IB_SUCCESS) {
		fprintf(stderr, "Failed to get port attributes: %s\n",
			ib_get_err_str(status));
		exit(-1);
	}

	for (i = 0; i < num_ports; i++) {
		if (attr_array[i].link_state == IB_LINK_ACTIVE)
			port_guid = attr_array[i].port_guid;
	}

	if (port_guid == (uint64_t)-1) {
		fprintf(stderr, "Failed to find active port, check port status with \"ibstat\"\n");
		exit(-1);
	}

	bind_handle = osmv_bind_sa(vendor, &mad_pool, port_guid);

	if (bind_handle == OSM_BIND_INVALID_HANDLE) {
		fprintf(stderr, "Failed to bind to SA\n");
		exit(-1);
	}
	return (bind_handle);
}

static void
clean_up(void)
{
	osm_mad_pool_destroy(&mad_pool);
	osm_vendor_delete(&vendor);
}

static void
usage(void)
{
	fprintf(stderr, "Usage: %s [-h -d -P -N -L -G -s -g -m][<name>]\n", argv0);
	fprintf(stderr, "   Queries node records by default\n");
	fprintf(stderr, "   -d enable debugging\n");
	fprintf(stderr, "   -P get PathRecord info\n");
	fprintf(stderr, "   -N get NodeRecord info\n");
	fprintf(stderr, "   -L Return just the Lid of the name specified\n");
	fprintf(stderr, "   -G Return just the Guid of the name specified\n");
	fprintf(stderr, "   -s Return the PortInfoRecords with isSM capability mask bit on\n");
	fprintf(stderr, "   -g get multicast group info\n");
	fprintf(stderr, "   -m get multicast member info\n");
	exit(-1);
}

int
main(int argc, char **argv)
{
	int                ch = 0;
	int                members = 0;
	osm_bind_handle_t  bind_handle;

	static char const str_opts[] = "PNLGsgmdh";
	static const struct option long_opts [] = {
	   {"P", 0, 0, 'P'},
	   {"N", 0, 0, 'N'},
	   {"L", 0, 0, 'L'},
	   {"G", 0, 0, 'G'},
	   {"s", 0, 0, 's'},
	   {"g", 0, 0, 'g'},
	   {"m", 0, 0, 'm'},
	   {"d", 0, 0, 'd'},
	   {"help", 0, 0, 'h'},
	   { }
	};

	argv0 = argv[0];

	while ((ch = getopt_long(argc, argv, str_opts, long_opts, NULL)) != -1) {
		switch (ch) {
		case 'P':
			query_type = IB_MAD_ATTR_PATH_RECORD;
			break;
		case 'N':
			query_type = IB_MAD_ATTR_NODE_RECORD;
			break;
		case 'L':
			node_print_desc = LID_ONLY;
			break;
		case 'G':
			node_print_desc = GUID_ONLY;
			break;
		case 's':
			query_type = IB_MAD_ATTR_PORTINFO_RECORD;
			break;
		case 'g':
			query_type = IB_MAD_ATTR_MCMEMBER_RECORD;
			break;
		case 'm':
			query_type = IB_MAD_ATTR_MCMEMBER_RECORD;
			members = 1;
			break;
		case 'd':
			osm_debug = 1;
			break;
		case 'h':
		default:
			usage();
		}
	}
	argc -= optind;
	argv += optind;

	if (argc)
		requested_name = argv[0];

	bind_handle = get_bind_handle();

	switch (query_type) {
	case IB_MAD_ATTR_NODE_RECORD:
		print_node_records(bind_handle);
		break;
	case IB_MAD_ATTR_PATH_RECORD:
		print_path_records(bind_handle);
		break;
	case IB_MAD_ATTR_PORTINFO_RECORD:
		print_portinfo_records(bind_handle);
		break;
	case IB_MAD_ATTR_MCMEMBER_RECORD:
		print_multicast_group_records(bind_handle, members);
		break;
	default:
		fprintf(stderr, "Unknown query type %d\n", query_type);
		break;
	}

	clean_up();
	return (0);
}
